package edu.northwestern.at.morphadorner.tools.lgparser;

/*  Please see the license information at the end of this file. */

import java.util.StringTokenizer;
import net.sf.jlinkgrammar.* ;
import edu.northwestern.at.utils.StringUtils;

/** Link grammar parser driver.
 *
 *  <p>
 *  Usage:
 *  </p>
 *  <blockquote>
 *  <pre>
 *  java edu.northwestern.at.morphadorner.tools.lgparser.LGParser lgparserdatadirectory "sentence text to parse"
 *  </pre>
 *  </blockquote>
 *  <p>
 *  where <strong>lgparserdatadirectory</strong> is a directory containing link
 *  grammar parser data files and "sentence to parse" is the text of the
 *  sentence to parse.
 *  </p>
 */

public class LGParser
{
    /** Parser dictionary. */

    protected static Dictionary dictionary ;

    /** Parser options. */

    protected ParseOptions parseOptions ;

    /** Sentence to parse. */

    protected Sentence sentence ;

    /** Data file directory. */

    protected static String defaultDataDirectory    = "data/lgparser";

    /** Create a linkage grammar parser. */

    public LGParser()
    {
        this( defaultDataDirectory , 100 , 1 );
    }

    /** Create a linkage grammar parser.
     *
     *  @param  dataDirectory   The data directory.
     */

    public LGParser( String dataDirectory )
    {
        this( dataDirectory , 100 , 1 );
    }

    /** Create a linkage grammar parser.
     *
     *  @param  maxLinkage      Maximum number of linkages.
     *  @param  maxParseTime    Maximum parse time in seconds.
     */

    public LGParser( int maxLinkage , int maxParseTime )
    {
        this( defaultDataDirectory , maxLinkage , maxParseTime );
    }

    /** Create a linkage grammar parser.
     *
     *  @param  dataDirectory   The data directory.
     *  @param  maxLinkage      Maximum number of linkages.
     *  @param  maxParseTime    Maximum parse time in seconds.
     */

    public LGParser
    (
        String dataDirectory ,
        int maxLinkage ,
        int maxParseTime
    )
    {
        parseOptions = new ParseOptions();

        parseOptions.parse_options_set_short_length( 10 );
        parseOptions.parse_options_set_max_null_count( 10 );
        parseOptions.parse_options_set_linkage_limit( maxLinkage );

        if ( dictionary == null )
        {
            try
            {
                dictionary =
                    new Dictionary
                    (
                        parseOptions ,
                        dataDirectory + "/4.0.dict" ,
                        "4.0.knowledge" ,
                        "4.0.constituent-knowledge" ,
                        "4.0.affix"
                    );
            }
            catch ( Exception e )
            {
                e.printStackTrace();
            }
        }
    }

    /** Get a linkage from a parsed sentence.
     *
     *  @param  index   The index of the linkage to return.
     *
     *  @return         The linkage at the specified index.
     */

    public Linkage getLinkage( int index )
    {
        return new Linkage( index , sentence , parseOptions );
    }

    /** Parse sentence text.
     *
     *  @param  s   Sentence text to parse.
     *
     *  @return     Parse sentence.
     */

    public Sentence parse( String s )
    {
        sentence = new Sentence( s , dictionary , parseOptions );

        sentence.sentence_parse( parseOptions );

        return sentence ;
    }

    /** Main program.
     *
     *  @param  args    Command line arguments.
     */

    public static void main( String[] args )
    {
        if ( args.length > 1 )
        {
                                //  Get link grammar data directory.

            String dataDirectory    = args[ 0 ];

                                //  Get sentence text to parse.

            String textToParse      = args[ 1 ] ;

                                //  Create new parser.

            LGParser parser = new LGParser( dataDirectory );

                                //  Parse sentence text.

            Sentence sentence = parser.parse( textToParse );

                                //  Display linkages, if any found.

            if ( sentence.sentence_num_linkages_found() < 1 )
            {
                System.out.println( "No linkage was found." );
            }
            else
            {
                Linkage link = parser.getLinkage( 0 );

                System.out.println( link.linkage_print_diagram() );

                System.out.println
                (
                    fixOutput( link.linkage_print_links_and_domains() )
                );

                System.out.println
                (
                    link.linkage_print_constituent_tree( 1 )
                );
            }
        }
    }

    /** Fix the output generated by the parser for presentation.
     *
     *  @param  s   Output generated by parser.
     *
     *  @return     Corrected output.
     */

    protected static String fixOutput( String s )
    {
                                //  Always six columns of output.

        int[] colWidths = new int[ 6 ];

                                //  Holds maximum width of each column.

        for ( int i =0 ; i < colWidths.length ; i++ )
        {
            colWidths[ i ]  = 0;
        }
                                //  Split output into lines.

        String[] lines  = s.split( "\n" );

                                //  For each line of output ...

        for ( int i = 0 ; i < lines.length ; i++ )
        {
                                //  Get next line.

            String line = lines[ i ];

                                //  If line is not empty ...

            if ( !line.equals( "\n" ) )
            {
                                //  Split line into columns and
                                //  compute width of each column.

                StringTokenizer tokenizer   =
                    new StringTokenizer( line );

                int j   = 0;

                if ( line.charAt( 0 ) == ' ' )
                {
                    j++;
                }

                while ( tokenizer.hasMoreTokens() )
                {
                    String token    = tokenizer.nextToken().trim();

                                //  Remember the maximum width of each
                                //  column so far.

                    colWidths[ j ]  =
                        Math.max( colWidths[ j ] , token.length() );

                    j++;
                }
            }
        }
                                //  Add space padding to each output column.

        for ( int i = 0 ; i < colWidths.length ; i++ )
        {
            colWidths[ i ] += 2;
        }
                                //  Reconstruct output, using padded
                                //  maximum column widths we just
                                //  determined.  The corrected output
                                //  is accumulated in a string buffer.

        StringBuffer sb = new StringBuffer();

        for ( int i = 0 ; i < lines.length ; i++ )
        {
            String line = lines[ i ];

            if ( !line.equals( "\n" ) )
            {
                StringTokenizer tokenizer   =
                    new StringTokenizer( line );

                int j   = 0;

                if ( line.charAt( 0 ) == ' ' )
                {
                    sb.append( StringUtils.dupl( " " ,  colWidths[ j++ ] ) );
                }

                while ( tokenizer.hasMoreTokens() )
                {
                    String token    = tokenizer.nextToken();

                    sb.append( StringUtils.rpad( token , colWidths[ j++ ] ) );
                }

                sb.append( "\n" );
            }
        }
                                //  Return corrected output.

        return sb.toString();
    }
}

/*
Copyright (c) 2008, 2013 by Northwestern University.
All rights reserved.

Developed by:
   Academic and Research Technologies
   Northwestern University
   http://www.it.northwestern.edu/about/departments/at/

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal with the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimers.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimers in the documentation and/or other materials provided
      with the distribution.

    * Neither the names of Academic and Research Technologies,
      Northwestern University, nor the names of its contributors may be
      used to endorse or promote products derived from this Software
      without specific prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE CONTRIBUTORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS WITH THE SOFTWARE.
*/



